﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using Domain.Services;
    using Hims.Api.Models;
    using Hims.Domain.Entities;
    using Hims.Domain.EntityModels;
    using Hims.Domain.Repositories.UnitOfWork;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Microsoft.CodeAnalysis.CSharp;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.Library.Enums;
    using Shared.UserModels.Filters;
    using Utilities;

    /// <inheritdoc />
    [Authorize]
    [Route("api/charge-category")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class ChargeCategoryController : BaseController
    {
        /// <summary>
        /// The coupon services.
        /// </summary>
        private readonly IChargeCategoryService chargeCategoryService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;
        /// <summary>
        /// The unit of work.
        /// </summary>
        private readonly IUnitOfWork unitOfWork;

        /// <inheritdoc />
        public ChargeCategoryController(IChargeCategoryService chargeCategoryService, IAuditLogService auditLogServices, IUnitOfWork unitOfWork)
        {
            this.chargeCategoryService = chargeCategoryService;
            this.auditLogServices = auditLogServices;
            this.unitOfWork= unitOfWork;
            
        }

        /// <summary>
        /// The fetch coupons.
        /// </summary>
        /// <param name="model">
        /// The coupon filter model.
        /// </param>
        /// <returns>
        /// The list of coupons.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of coupons.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<ChargeTypesModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody] ChargeCategoryModel model)
        {
            model = (ChargeCategoryModel)EmptyFilter.Handler(model);
            var charge = await this.chargeCategoryService.FetchAsync(model);
            return charge == null ? this.ServerError() : this.Success(charge);
        }


        /// <summary>
        /// The add coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Charge added successfully.
        /// - 409 - Charge already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] ChargeCategoryModel model, [FromHeader] LocationHeader header)
        {
            model = (ChargeCategoryModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            var response = await this.chargeCategoryService.AddAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given Charge Category has already been exists with us.");
                case 0:
                    return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.RoomCharge,
                LogFrom = (short)model.LoginRoleId,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $" {model.CreatedByName} has added Room Charge Category - {model.ChargeCategoryName} on {DateTime.UtcNow.AddMinutes(330)}",
                LocationId = Convert.ToInt32(header.LocationId)

            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("Charge Type has been added successfully.");
        }

        /// <summary>
        /// The update charge.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - charge updated successfully.
        /// - 409 - Charge already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] ChargeCategoryModel model, [FromHeader] LocationHeader header)
        {

            model = (ChargeCategoryModel)EmptyFilter.Handler(model);

            var response = await this.chargeCategoryService.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given Charge Category has already been exists with us.");
                case 0:
                    return this.ServerError();
            }

            var auditLog = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.RoomCharge,
                LogDate = DateTime.Now,
                LogFrom = short.Parse(model.LoginRoleId.ToString()),
                LogDescription = $"{model.CreatedByName} has updated <b>Charge Type</b> of <strong>{model.ChargeCategoryName}</strong> successfully.",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLog);

            return this.Success("Charge Type has been updated successfully.");


        }

        /// <summary>
        /// The delete charge type.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - charge type deleted successfully.
        /// - 409 - charge type can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("modify-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] ChargeCategoryModel model, [FromHeader] LocationHeader header)
        {

            model = (ChargeCategoryModel)EmptyFilter.Handler(model);

            var response = await this.chargeCategoryService.ModifyStatusAsync(model);
            if (response == 0)
            {
                return this.ServerError();
            }

            var auditLog = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.RoomCharge,
                LogDate = DateTime.Now,
                LogFrom = (short)model.LoginRoleId,
                LogDescription = $@"<b>{model.CreatedByName}</b> has <b>{((bool)model.Active ? "Activated" : "Deactivated")}</b> the Charge Type <b>{model.ChargeCategoryName}</b> successfully",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLog);

            return this.Success(response);
        }

        /// <summary>
        /// Gets the charge category masters asynchronous.
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        [Authorize]
        [Route("fetch-location-charges")] //useless
        public async Task<ActionResult> GetChargeCategoryMastersAsync()
        {
            var resources = await this.chargeCategoryService.FetchChargeCategoryMastersAsync();
            return this.Success(resources);
        }


        /// <summary>
        /// The fetch room charges.
        /// </summary>       
        /// <returns>
        /// The list of room categories charges.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of coupons.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch-room-charges")]
        [ProducesResponseType(typeof(List<ChargeTypesModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchRoomChargesAsync([FromBody] ChargeCategoryModel model)
        {
            model = (ChargeCategoryModel)EmptyFilter.Handler(model);
            var charge = await this.chargeCategoryService.FetchRoomChargeAsync(model);
            return charge == null ? this.ServerError() : this.Success(charge);
        }

        /// <summary>
        /// The add coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Charge added successfully.
        /// - 409 - Charge already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("insert-category-charges")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> InsertAsync([FromBody] ChargeCategoryModel model, [FromHeader] LocationHeader header)
        {
            model = (ChargeCategoryModel)EmptyFilter.Handler(model);
            var response = await this.chargeCategoryService.InsertChargetoCategoryAsync(model);
            var categoryDetails = this.unitOfWork.ChargeCategorys.Find(s => s.ChargeCategoryId == model.ChargeCategoryId);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given Location has already exists with Given Category.");
                case 0:
                    return this.ServerError();
            }
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.RoomCharge,
                LogFrom = (short)model.LoginRoleId,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $"<b> {model.CreatedByName}</b> has added Room Charge Category -<b> {categoryDetails.ChargeCategoryName}</b> on {DateTime.UtcNow.AddMinutes(330)}",
                LocationId = Convert.ToInt32(header.LocationId)

            };
            await this.auditLogServices.LogAsync(auditLogModel);
            

            return this.Success("Charge has been added successfully.");
        }

        /// <summary>
        /// The add coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Charge added successfully.
        /// - 409 - Charge already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("update-category-charges")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateChargeAsync([FromBody] ChargeCategoryModel model, [FromHeader] LocationHeader header)
        {
            model = (ChargeCategoryModel)EmptyFilter.Handler(model);
            var response = await this.chargeCategoryService.UpdateChargetoCategoryAsync(model);
            var categoryDetails = this.unitOfWork.ChargeCategorys.Find(s => s.ChargeCategoryId == model.ChargeCategoryId);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given Location has already exists with Given Category.");
                case 0:
                    return this.ServerError();
            }
            var auditLog = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.RoomCharge,
                LogDate = DateTime.Now,
                LogFrom = short.Parse(model.LoginRoleId.ToString()),
                LogDescription = $"<b>{model.CreatedByName}</b> has updated <b>Charge Type</b> of <strong>{categoryDetails.ChargeCategoryName}</strong> successfully.",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLog);
            return this.Success("Charge has been Updated successfully.");
        }

        /// <summary>
        /// Sets the default charge asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("set-default-category-charges")]
        public async Task<ActionResult> SetDefaultChargeAsync([FromBody] ChargeCategoryModel model)
        {
            model = (ChargeCategoryModel)EmptyFilter.Handler(model);
            var response = await this.chargeCategoryService.MakeDefaultChargeCategoryAsync(model);
            return this.Success(response);
        }
        [HttpPost]
        [Route("modify-roomstatus")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyRooms([FromBody] ChargeCategoryModel model, [FromHeader] LocationHeader header)
        {

            model = (ChargeCategoryModel)EmptyFilter.Handler(model);

            var response = await this.chargeCategoryService.ModifyStatusRoomAsync((model));
            if (response == 0)
            {
                return this.ServerError();
            }

            var auditLog = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.RoomCharge,
                LogDate = DateTime.Now,
                LogFrom = (short)model.LoginRoleId,
               
                LogDescription = $@"<b>{model.CreatedByName}</b> has {((bool)model.Active ? "Activated" : "Deactivated")} Room Charge Type <b> {model.ChargeCategoryName}</b>  successfully",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLog);

            return this.Success("Charge Category has been updated the status.");
        }
    }

}
